// EditableList.java

package NormsTools;

import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
//import java.util.Date;      // For testing

/** Display a list of strings and allow Addition, deletion and moving of items in
* the list.  Requires user to implement EditableListAdder interface.
*/
public class EditableList extends Dialog implements ActionListener {

    private Panel       p1 = new Panel(),
                        bPanel = new Panel();

    private Button      addB = new Button("Add"),
                        moveUpB = new Button("Move up"),
                        moveDownB = new Button("Move down"),
                        deleteB = new Button("Remove"),
                        doneB = new Button("Done"),
                        cancelB = new Button("Quit");

    private GridBagConstraints gbc;
    private Image   saveIcon = null;
    private Color   saveColor = null;

    private EditableListAdder ELA;
    private String[]    theList = null;
    private java.awt.List shownList; 

    final static int    NONE_SEL = -1;      // indicate nothing selected
    private      int    selIdx = NONE_SEL;

    private boolean changeMade = false;     // Remember

    // Constructors ------------------------------------------------------
    /**  Create a list using items in String[]
    */
    public EditableList(Frame f, String title, String[] items, 
                                        EditableListAdder ela) {
        super(f, title, true);
        ELA = ela;
        finishUp(title, items);
    }
    /** Create a list using items in a Vector 
    */
    public EditableList(Frame f, String title, Vector v, EditableListAdder ela) {
        super(f, title, true);
        ELA = ela;
        String[] vs = new String[v.size()];
        v.copyInto((Object[])vs);
        finishUp(title, vs);
    }
    // end Constructors

    //-------------------------------------------------------------
    // common routine to finish GUI etc
    private void finishUp(String title, String[] items) {
        setTitle(title);
        setSize(600, 400);    // This will be overriden by pack()
        setLayout(new BorderLayout());
        // trap user closing window
        addWindowListener(new WindowAdapter() {
          public void windowClosing(WindowEvent we) {
            exitPgm();
          }
        });    
        
        p1.setLayout(new BorderLayout());
        // Build description area at top of display
        p1.add(new Label(title), BorderLayout.NORTH);

        int nbrID = items.length;       // get number in table

        // . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .
        // Now build listbox, fill with the strings and add to Frame:
        int nbrRows = Math.min(10, nbrID+2); // #+2 to 10
        shownList = new java.awt.List(nbrRows, false);
        for (int i = 0; i < nbrID; i++) {
          shownList.add(items[i]);
        } 
        // Trap list selections HERE:
        shownList.addItemListener(new ItemListener() {
          public void itemStateChanged(ItemEvent ie) {
            // First item selected will just be saved, the second selection
            // will cause first item to be moved before the second item
            Object target = (java.awt.List)ie.getSource();
            if (target == shownList) {
              selIdx = shownList.getSelectedIndex();  // save index
              if (selIdx == NONE_SEL) return;     // exit if nothing selected

              //OK there is an item selected:
              // Enable some of the buttons depending on ...
              if (shownList.getItemCount() > 1)
                  setMoveButtons();
              deleteB.setEnabled(true);

            }else {
                System.err.println("Unknown item event " + ie);
            }
          }  
        });   // end ItemListener
        shownList.requestFocus();
        p1.add(shownList, BorderLayout.CENTER);

        // . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .
        // Build a panel with buttons vertically
        bPanel = new Panel();
        GridBagLayout gbl = new GridBagLayout();
        bPanel.setLayout(gbl);
        gbc = new GridBagConstraints();
        gbc.weightx = 100;
        gbc.weighty = 100;
        gbc.anchor = GridBagConstraints.WEST;       // left adjust
        gbc.fill = GridBagConstraints.HORIZONTAL;  // vs BOTH????
        gbc.insets = new Insets(2, 0, 2, 0);
        int bRow = 0;

        addGBC(addB, 0, bRow++, 1, 1);
        addGBC(moveUpB, 0, bRow++, 1, 1);
        moveUpB.setEnabled(false);
        addGBC(moveDownB, 0, bRow++, 1, 1);
        moveDownB.setEnabled(false);
        addGBC(deleteB, 0, bRow++, 1, 1);
        deleteB.setEnabled(false);
        addGBC(doneB, 0, bRow++, 1, 1);
        doneB.setEnabled(true);         // How to set???
        addGBC(cancelB, 0, bRow++, 1, 1);
        p1.add(bPanel, BorderLayout.EAST);    
        add(p1, BorderLayout.NORTH);

        // Set color if value given earlier
        if (saveColor != null)
            setBackground(saveColor);

        pack();               // Compute min size and resize

        // Center frame
        Dimension ss = Toolkit.getDefaultToolkit().getScreenSize();
        setLocation((ss.width-getBounds().width)/2, 
                    (ss.height-getBounds().height)/2);  

        setVisible(true);
    } // end finishUp()

    //-----------------------------------------------------
    // Utility for showList() - places buttons in panel
    private void addGBC(Button b, int x, int y, int w, int h) {
        gbc.gridx = x;
        gbc.gridy = y;
        gbc.gridwidth = w;
        gbc.gridheight = h;
        bPanel.add(b, gbc);
        b.addActionListener(this);
    }  // end addGBC()

    //-----------------------------------------------------
    private void exitPgm() {
        setVisible(false);
        dispose();
//        System.out.println("Exiting pgm");
    }  // end exitPgm()

    //-----------------------------------------------------
    public void actionPerformed(ActionEvent ae) {
        Object src = ae.getSource();

        if(src == addB) {
            // Get the item to add
            String newOne = ELA.addToList();
            if (newOne == null || newOne.equals(""))
                return;         // ignore 
            if (selIdx != NONE_SEL) {
                // add after selected item
                shownList.addItem(newOne, selIdx+1); // add after
            } else
                shownList.addItem(newOne, 0);    // put first
            setMoveButtons();
            changeMade = true;          // remember

        } else if(src == moveUpB) {
            if (selIdx > 0) {
                String item = shownList.getItem(selIdx);
                shownList.remove(selIdx);
                selIdx -= 1;            // move up 1
                shownList.add(item, selIdx);
                shownList.select(selIdx);
                setMoveButtons();
                changeMade = true;      // remember
            }

        } else if(src == moveDownB) {
            if (selIdx != NONE_SEL && selIdx < shownList.getItemCount()-1) {
                String item = shownList.getItem(selIdx);
                shownList.add(item, selIdx+2);       // add after next
                shownList.remove(selIdx);
                selIdx++;               // advance to new position
                shownList.select(selIdx);
                setMoveButtons();
                changeMade = true;      // remember
            }

        } else if(src == deleteB) {
            if (selIdx != NONE_SEL) {
                shownList.remove(selIdx);
                selIdx = NONE_SEL;
                changeMade = true;      // remember
            }
        
        }else if(src == doneB) {
            exitPgm();

        }else if(src == cancelB) {
            changeMade = false;         // Forget it!
            exitPgm();

        }else { 
            System.err.println("a..P.. Unknown source " + src);
        }
    }  // end actionPerformed()

    // Set buttons on if not at boundary
    private void setMoveButtons() {
        moveDownB.setEnabled(selIdx != shownList.getItemCount()-1); // off at bottom
        moveUpB.setEnabled(selIdx != 0);     // turn off at top
    }

    //-----------------------------------------------------------
    /** Returns true if a change was made to the list
    */
    public boolean changed() {
        return changeMade;
    }
    /** Return the new updated list in a String[]
    */
    public String[] getList() {
        return shownList.getItems();
    }
    /** Return the updated list in a Vector
    */
    public Vector  getListInVector() {
        theList = shownList.getItems();
        Vector v = new Vector(theList.length);
        for(int i = 0; i < theList.length; i++)
            v.addElement(theList[i]);
        return v;
    } // end getListInVector()

    //----------------------------------------------------------------------
    // For testing
/*    public static void main(String[] args) {
        String[] list = new String[] {"First", "Second", "Third", "Fourth"};
        // Inner class for adder
        class EL implements EditableListAdder {
            public String addToList() {
                return "New Item " + new Date();
            }
        } // end class
        EditableList el = new EditableList(new Frame(), "Testing", list, new EL());
        System.out.println("Rtnd from constructor");
        System.out.println("Results: " + el.getListInVector().toString());
        System.exit(0);
    }  // end main()  for testing */
}  // end EditableList