// DoubleInputField.java - double input field with Label

package NormsTools;

import java.awt.*;
import java.awt.event.*;
import java.text.*;

//////////////////////////////////////////////////////////////////
// Define class with input field for double

public class DoubleInputField extends Panel implements FocusListener {
    MyLabel     lbl;
    TextField   tf;
    double      value;

    int             lastPos;
    Frame           errF = null;
    DecimalFormat   df = null;
    ErrDialog       ed = null;
    boolean         setTextAllowed = true;

    // Range checking values:
    double      maxValue = 1000000;
    double      minValue = 0;
    FlowLayout  fl = new FlowLayout(FlowLayout.LEFT);

    public DoubleInputField(String label, int size, double defval) {
        fl.setHgap(0);
        setLayout(fl);
        lbl = new MyLabel(label, Label.LEFT);
        add(lbl);
        if (size < 0)
            tf = new TextField("" + defval);
        else
            tf = new TextField("" + defval, size);
        add(tf);

        tf.addFocusListener(this);
        tf.addKeyListener(new KeyAdapter() {
            public void keyTyped(KeyEvent ke) {
                char ch = ke.getKeyChar();
                // Don't allow invalid chars.  What about negative values?
                if (!(('0' <= ch && ch <= '9')
                        || ch == '.'
                        || Character.isISOControl(ch)))
                    ke.consume();       // eat invalid chars
                else
                   lastPos = tf.getCaretPosition();  // save location
            }
        });
        value = defval;
    } // end Constructor


    //--------------------------------------
    // Property handlers
    public double getValue() {
        checkValue();
        return value;
    }

    public void setValue(double v) {
        value = v;
        if (!setTextAllowed)        // keep text field quiet
            return;

        if (df != null)
            tf.setText(df.format(value));
        else
            tf.setText("" + v);
    }

    public void enableSetText(boolean b) {
        setTextAllowed = b;
    }

    public void ckValid() {
        checkValue();
    }

    public void setMinMax(double min, double max) {
        if (min < max) {
            minValue = min;
            maxValue = max;
        } else {
            if (errF != null)
                ed = new ErrDialog(errF, lbl.getText() 
                            + "Invalid min:max values for " + lbl.getText()); 
            else
                System.err.println("Invalid min:max values for " 
                                    + lbl.getText());
        }
    }

    public void setErrFrame(Frame f) {
        errF = f;
    }

    public void setDecimalFormat(DecimalFormat df) {
        // Need to reformat according to new formater
        if (df == null) {
        }
        this.df = df;
    }
   
    public String toString() {
        if (df != null)
            return df.format(value); 
        else
            return Double.toString(value);
    }

    //-------------------------------------------------------------
    
    // Get/check integer value from Text field
    private void checkValue() {
        // Skip checking if empty field
        if (tf.getText().trim().equals(""))
            return;

        try {
            double aValue = 0;
            if (df == null)
                aValue = Double.valueOf(tf.getText().trim()).doubleValue();
            else
                aValue = df.parse(tf.getText().trim()).doubleValue();

            if (aValue > maxValue || aValue < minValue) {
                if (errF != null) {
                    ed = new ErrDialog(errF, lbl.getText() + " out of range: " 
                        + minValue + ":" + maxValue);
                } else    
                    System.err.println(lbl.getText() + " out of range: " 
                                    + minValue + ":" + maxValue);
                tf.requestFocus();      // position cursor in textfield
            } else
                value = aValue;         // save if OK

        }catch(NumberFormatException ne) {
            System.err.println(ne);
            tf.setText("" + value);
            tf.setCaretPosition(lastPos);
        }catch(ParseException pe) {
            System.err.println(pe);
            tf.setText("" + value);
            tf.setCaretPosition(lastPos);
        }
    } // end checkValue()

    public void focusGained(FocusEvent fe) {}
            
    public void focusLost(FocusEvent fe) {
        if (fe.getComponent() == tf)
            if (tf.getText().trim().equals("")) {
                if (errF != null) {
                    ed = new ErrDialog(errF, lbl.getText() 
                            + " missing value"); 
                }    
                else
                    System.err.println(lbl.getText() + " missing value");
                tf.requestFocus();      // position cursor in textfield
            }else
                checkValue();
         else {
            System.out.println("unknwn focusLost: " + fe);
        }
    }  // end  focusLost()

    //--------------------------------------------------
    // Following for debugging.  Comment out when done
/*    public static void main(String[] args) {
        TestFrame f = new TestFrame();
        f.add(new DoubleInputField("Testing", 4, 100));
        f.show();
    } // end main()  */

}  // end class DoubleInputField
