//CLArgs.java - CommandLine args parser
// Commandline arg syntax: options start with a - and are 1 or more chars long
//      case sensitive. Unambiguous substrings allowed
//      options with following values are defined with an ending =
//      options are before the positional args


package NormsTools;

import java.util.Hashtable;

public class CLArgs {

    public final static String    OptPrefix = "-";    // Option's leading character
    public final static String    ValueFollows = "="; // Signifies that a value follows

    // If defined as "Filename="  then use as  -File aFile.txt

	private Hashtable ht = new Hashtable();		// save args here
    // Following two arrays are used in parallel
    private String[]    optDefs;                // Holds option definitions
    private int[]       minLen;                 // Minimum allowed length
    private String[]    pArgs;                  // Holds positional args(if any)

    final boolean   debug = false;      // For debug output


	// Constructor - receives array of option definitions
	public CLArgs(String[] defs) {
        optDefs = new String[defs.length];        // make a new home
        minLen = new int[defs.length];
        System.arraycopy(defs, 0, optDefs, 0, defs.length);
        // Look for minimum lengths to use
        // For example:
        //    xxx   2
        //    xyy   3
        //    xyz   3
        for(int i =0; i < defs.length; i++) {
            for(int j = i+1; j < defs.length; j++) {
                if(optDefs[i].equals(optDefs[j])) {
                    throw new IllegalArgumentException("Duplicate option " 
                                                                + optDefs[i]);
                }
                // Get index to first different character
                int ix = findDiff(optDefs[i], optDefs[j]);
                int iy = ix + 1;        // Get length of
                // Test for case of same keyword & keyword= such as DEF and DEF=
                if ((ix == optDefs[i].length() && optDefs[j].length() == iy
                      && optDefs[j].endsWith(ValueFollows))
                    || (ix == optDefs[j].length() && optDefs[i].length() == iy
                      && optDefs[i].endsWith(ValueFollows)) ) {
                    throw new IllegalArgumentException("Ambiguous options " 
                                    + optDefs[i] + " & " +  optDefs[j]);
                }
                if (debug)
                    System.out.println("findDiff was " + ix  + " for " + optDefs[i] 
                                                    + " & "  + optDefs[j]);
                // Now save minimum lengths
                if(iy > minLen[i])
                    minLen[i] = iy;
                if(iy > minLen[j])
                    minLen[j] = iy;
            }
        } // end editting option definitions  outter loop
	} // end Constructor

    // Find index of first different character in two strings
    private int findDiff(String a, String b) {
        for(int i = 0; i < a.length() && i < b.length(); i++) {
            if (a.charAt(i) != b.charAt(i))
                return i;
        }
        //If shorter is a substring of longer, then return length of shorter
        return Math.min(a.length(), b.length());
    } // end findDiff

    // Return a string showing usage of args: 
    //    -opt1 -opt2 <opt2Value> -opt3 ...
    public String usage() {
        String retValue = "";
        for (int i=0; i < optDefs.length; i++) {
            if(optDefs[i].endsWith(ValueFollows)) {
                String sub1 = optDefs[i].substring(0, optDefs[i].length()-1);
                retValue += OptPrefix + sub1 + " " + "<" + sub1 + "Value> ";
            }else {
                retValue += OptPrefix + optDefs[i] + " ";
            }
        } // end for
        return retValue;
    }   // end usage()

    // -------------------------------------------------------------
    // Parse the args
    public void set(String[] args)  throws IllegalArgumentException {
        for(int i = 0; i < args.length; i++) {
            // Check if arg is an option - Starts with -
            if (args[i].startsWith(OptPrefix)) {
                if (args[i].length() < 1) {
                    System.err.println("Valid options: " + usage());
                    throw new IllegalArgumentException("Missing option " + args[i]);
                }

                // Check if valid option
                String woDash = args[i].substring(1);
                if (debug)
                    System.out.println("woDash " + woDash);
                boolean fndOption = false;
                for(int j = 0; j < optDefs.length; j++) {
                    if (optDefs[j].startsWith(woDash) 
                          && woDash.length() >= minLen[j]) {
                        // Found one - Check if has value following it
                        if (optDefs[j].endsWith(ValueFollows)) {
                            if ((i+1) < args.length) {
                                checkOnly(optDefs[j], i);
                                ht.put(optDefs[j], args[++i]);
                                if (debug)
                                    System.out.println("Saving " + optDefs[j] 
                                                            + " with " + args[i]);
                            }else {
                                System.err.println("Valid options: " + usage());
                                throw new IllegalArgumentException(optDefs[j] 
                                    + " requires a value");
                            }
                        }else {    
                            checkOnly(optDefs[j], i);
                            ht.put(optDefs[j],"");
                            if (debug)
                                System.out.println("Saving " + optDefs[j] 
                                                            + " for " + args[i]);
                        }
                        fndOption = true;       // no error now
                        break;          // done with this arg
                    }
                } // end for j searching for options

                // Report error if this arg was not found
                if (!fndOption) {
                    System.err.println("Valid options: " + usage());
                    throw new IllegalArgumentException("Unknown option: " + args[i]);
                }
 
            }else {
                // This one doesn't start with a -, rest of args are positional
                pArgs = new String[args.length-i];
                System.arraycopy(args, i, pArgs, 0, pArgs.length);
                break;          // Done exit loop
            }
        } // end for i scanning args
    }  // end set()

    // Check that xx has not already been found and saved
    private void checkOnly(String xx, int i) {
        if (ht.containsKey(xx)) {
            throw new IllegalArgumentException("Reused option: " + xx + " at " + i);
        }
    }

	// Tell user if we found an arg
	public boolean had(String key) {
		return ht.containsKey(key);
	}

	// Return value of an arg
	public String get(String key) {
		return (String)ht.get(key);
	}

    // Tell user if there were positional args
    public boolean hadPosits() {
        return pArgs != null;
    }
    // Return value of a positional arg
    public String getPosit(int i) {
        return pArgs[i];
    }

    //--------------------------------------------------------------------
	// Following for testing. Comment out when done ---------------------------
	public static void main(String[] args) {
        // Define some local names
        final String fn = "Filename=";
        final String fnss = "FileX";
        final String fx = "Fixit";
        final String debug = "Debug";

        // Define our options
        CLArgs cla = new CLArgs(new String[] {debug , fn, fx, fnss,
//                    "DEF=", "DEF",  // ambiguous
                    "xxx", "xyy", "xyz"});

        // Here's the command line for testing:
        if (args.length == 0) {
            args = new String[] {"-De", "-Filen", "afile.txt", "-Fix",
//                                    "-Debug",    // duplicate
                                    "-xx", "-xyy", 
//                                    "-bad",      // unknown
                                    "FIRST_ONE"};  // Positional at end
        }

        cla.set(args);
        System.out.println("Usage: " + cla.usage());

        if(cla.had(debug))
            System.out.println("Had debug");
        if (cla.had(fn))
            System.out.println(fn + cla.get(fn));
        if (cla.had(fx))
            System.out.println("Had " + fx);
        if (cla.hadPosits()) 
            System.out.println("P[1] = " + cla.getPosit(0));
	} // end main() */
} // end class CLArgs
