// BigTextArea.java

/** Extends TextArea to allow very large outputs to be displayed.<BR>
*  Only displays a max size in the actual TextArea. Saves additional
* text in a Vector with methods to move up or down thru these saved texts.

  Changes:
    1 Nov 2001 - up one more parent to validate; Fixed getSelectedText() 

*/

package NormsTools;

import java.awt.TextArea;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Date;      // For testing

public class BigTextArea  extends TextArea       implements ActionListener    {

    private final boolean   debug = false;   // true for debug output 

    // -------------------------------------------------------------------------
    // Text will be appended to the super.TextArea until it holds maxSectionSz.
    // Then the text will be moved to the saveSections buffer and the TextArea
    // cleared.  The user can move up and down the text by calling goUp() or
    // goDown().  These methods will move the selected section to the TextArea 
    // for display. Tricky bit is with the last/partial section which will be
    // accumulated in a StringBuffer (svText) while the user is looking at 
    // previous sections and will go into the TextArea itself if that is where 
    // the user is veiwing.


    // -----------------------------------------------------------------------
    // Define a buffer to hold earlier sections of text 
    private Vector  savedSections = new Vector();

    private final int   NONE_SAVED = -1;

    private int     szCurrSection = 0,
                    currSection = 0,
                    maxSectionSz = 30000,    // Maximum number of bytes for TextArea
                    lastSectionSaved = NONE_SAVED;   // -1 = none saved yet
                    // NOTE: this should be the same as savedSections.size()-1

    // Define buffer to save text if currently viewing another section
    private StringBuffer svText;

    // Define flag to be used by append and getText() to tell getText() it is 
    // being called by the super.append() routine.
    private boolean     doingSuperAppend = false;

    // Save references to parent's controls
    private MenuItem    upMI,
                        downMI;

    private Button      upB,
                        downB;


    private BigTextAreaListener btaL = null;

    private String  svUpLbl = "",
                    svDownLbl = "";
    
    private final char nlChar = '\n';

    //Constructors --------------------------------------------------------------
    /** Constructors will be the same as TextArea with the exception that the
    * last int will be the maxSectionSz
    */
    public BigTextArea() {
        super();
        finishInit();
    }
    public BigTextArea(String str) {
        super(str);
        finishInit();
    }
    public BigTextArea(int ms) {
        super();
        maxSectionSz = ms;
        finishInit();
    }
    public BigTextArea(String str, int ms) {
        super(str);
        maxSectionSz = ms;
        finishInit();
    }
    public BigTextArea(int r, int c, int ms) {
        super(r, c);
        maxSectionSz = ms;
        finishInit();
    }
    public BigTextArea(String str, int r, int c, int ms) {
        super(str, r, c);
        maxSectionSz = ms;
        finishInit();
    }

    // Finish the constructor stuff
    private void finishInit() {
        svText = new StringBuffer(2*maxSectionSz);  // Allow enough room
    }
    //--- end of Constructors ----------------------------------------------------


    /** Set the maximum size of a section
    *  @param int - new maximum size
    */
    public void setMaxSectionSize(int ms) {
        maxSectionSz = ms;
        // Hopefully svText can expand ok now!!!
        svText.ensureCapacity(ms);
    }
    /** Returns the current maximum section size
    */
    public int  getMaxSectionSize() {
        return maxSectionSz;
    }

    public void addBigTextAreaListener(BigTextAreaListener btal) {
        btaL = btal;                    // save listener
    }

    // Save parent's control references
    public void setMenuItems(MenuItem upMI, MenuItem downMI) {
        this.upMI = upMI;
        this.downMI =  downMI;
    }

    public void setButtons(Button up, Button down, boolean addListener) {
        upB = up;
        svUpLbl = up.getLabel();
        downB = down;
        svDownLbl = down.getLabel();

        // Does user what us to handle the buttons?
        if (addListener) {
            upB.addActionListener(this);
            downB.addActionListener(this);
            upB.setBackground(Color.yellow);
            downB.setBackground(Color.yellow);
        }
    } // end setButtons()

    //----------------------------------------------------------------------
    // Send message to BigTextAreaListener - have created new section
    private void tellListener() {
        if (btaL != null) {
            btaL.addedSection(lastSectionSaved);
        }
    }  // end tellListener()

    //----------------------------------------------------------------------
    // Change display on the button labels by adding the current / last section
    // Call this method every time the current Section changes
    private void changeButtons() {
        if (btaL != null && upB != null) {
            // Now set buttons
            upB.setLabel(svUpLbl + " " + (currSection+1) 
              + (lastSectionSaved >= 0 ? ("/" + (lastSectionSaved+2)) : ""));
            upB.invalidate();
            if (upB.getParent() != null)
                upB.getParent().validate();
            downB.setLabel(svDownLbl + " " + (currSection+1)
              + (lastSectionSaved >= 0 ? ("/" + (lastSectionSaved+2)) : ""));
            downB.invalidate();
            if (downB.getParent() != null) {
                if (downB.getParent().getParent() != null) { // ADDED 11/01
                  downB.getParent().getParent().validate();
                }
                downB.getParent().validate();
            }
        }
    } // changeButtons()

    //-----------------------------------------------------------
    // Enable the controls
    // Up is towards the top or the first section
    // Down is towards the bottom or the last section.
    private void enableControls() {
        // Can go up if not at top
        if (upMI != null)
            upMI.setEnabled(currSection > 0);
        if (upB != null)
            upB.setEnabled(currSection > 0);
        // Can go down if not at bottom
        if (downMI != null)
            downMI.setEnabled(currSection <= lastSectionSaved);
        if (downB != null)
            downB.setEnabled(currSection <= lastSectionSaved);

        changeButtons();
    } // end enableControls()

    //-------------------------------------------------------------
    /** Move display up one section
    */
    public synchronized void goUp() {
        if (currSection == 0)
            return;                     // Can't go up???
        saveBottom();
        currSection--;
        super.setText((String)savedSections.elementAt(currSection));
        enableControls();               // keep settings current
    }
    /** Move display to top
    */
    public synchronized void goToTop() {
        saveBottom();
        currSection = 0;
        super.setText((String)savedSections.elementAt(currSection));
        enableControls();               // keep settings current
    }  // end goToTop()

    // Local method to save the bottom section before moving to another
    private void saveBottom() {
        if (currSection > lastSectionSaved) {
            // if at bottom, save what's being displayed in non display area
            String txt = super.getText();
            svText.setLength(0);        // Clear the buffer
            svText.append(txt);         // save partial in svText
            szCurrSection = txt.length();
        }
    }  // end saveBottom()

    /** Move display to selected section
    *        @param - Section id to go to
    */
    public synchronized void goToSection(int secId) {
        if(secId < 0 || secId > lastSectionSaved+1 || currSection == secId)
            return;                     // Out of bounds!!! or current already
        saveBottom();
        currSection = secId;
        super.setText((String)savedSections.elementAt(currSection));
        enableControls();               // keep settings current
    }  // end gotoSection()

    /** Move display down one section<BR>
    * at the last one, need to change appending mode
    */
    public synchronized void goDown() {
        if (currSection > lastSectionSaved)
            return;                     // Nowhere to go

        currSection++;                  // go to next
        enableControls();               // keep in sync
        // Are we at end?
        if (currSection > lastSectionSaved) {
            // At end get saved text into display
            super.setText(svText.toString());   // displayed saved text
            szCurrSection = svText.length();  // get nbr bytes
            svText.setLength(0);        // empty the buffer

        } else {
            super.setText((String)savedSections.elementAt(currSection));
        }

    } // end goDown()
    /** Go to the last display
    */
    public synchronized void goToBottom() {
        if (currSection == lastSectionSaved+1)
            return;                     // Exit if already there
        currSection = lastSectionSaved + 1; // set to be at bottom
        super.setText(svText.toString());   // displayed saved text
        szCurrSection = svText.length();    // get nbr bytes
        svText.setLength(0);            // empty the buffer
        enableControls();               // keep settings current
    } // end goToBottom()

    //-------------------------------------------------------------------
    /** Find a string in the BigTextArea
    *   @param - String to find
    *   @returns - true if found, false if not found
    */
    public boolean findStr(String str) {
        return true;
    }
    /** Find next copy of a string in the BigTextArea<BR>
    * Must have called findStr() first
    *   @returns - true if found, false if not found
    */
    public boolean findNextStr() {
        return true;
    }

    //-------------------------------------------------------------------------
    /** Set the selection in the TextArea
    *      @param - Starting col
    *      @param - Ending col
    */
    // Find where the starting col is, display that section and select the columns
    // NOTE: Can't span sections!!!
    // ADD CODE TO SAVE Selected locations to re-select when displayed again???
    public synchronized void select(int selStart, int selEnd) {
        if (debug) {
            System.out.println("select() start: " + selStart + " end:" + selEnd);
        }
        if (lastSectionSaved == NONE_SAVED) {
            super.select(selStart, selEnd);     // simple if nothing saved

        }else {
            // Find the section that has the selStart column
            int startCol = 0;           // Starting column for this section
            boolean fndSection = false;

            for (int i = 0; i < savedSections.size(); i++) {
                int secLength = ((String)savedSections.elementAt(i)).length();
                if (debug)
                    System.out.println("Section " + i + " length= " + secLength);
                // Check if startCol is in this section
                if (selStart >= startCol && selStart < startCol + secLength) {
                    goToSection(i);     // Load this section
                    fndSection = true;  // tell on exit
                    break;              // Exit with correct section loaded
                }
                startCol += secLength;  // Advance to start of next section
            } // end searching

            // If we fell out: either in currently displayed area
            // or in svText
            if (!fndSection) {
                goToBottom();           // restore from svText()
            }

            super.select(selStart - startCol, selEnd - startCol);
            if (debug)
                System.out.println(" super.select(" + (selStart - startCol)
                     + ", " + (selEnd - startCol) + ") in " + currSection
                     + " Original start=" + selStart + " end=" + selEnd);
        }
    }  // end select()

    //-------------------------------------------------------------------------
    /** Append text to display<BR>
    * If overflow, start saving in a StringBuffer
    */
    public synchronized void append(String txt)  {
        // Appending to current section?
        if (currSection > lastSectionSaved) {
            //for last section, add as usual to viewing TextArea.  Check if full
            // NOTE: NEED To test if txt is TOO BIG to add??? see breakUpStr()
            // if (txt.length() + szCurrSection > maxSectionSz ...
            // if (findMaxNL(txt, 0, maxSectionSz - szCurrSection)+1 == txt.length()
            // else have to breakUpStr and save to ...
//            Vector secs = breakUpStr(txt, maxSectionSz - szCurrSection);
//            savedSections.addElement(super.getText() + (String)secs.firstElement());

            doingSuperAppend = true;    // tell getText() who's calling
            super.append(txt);         //**** ???? this is having problems???
            doingSuperAppend = false;
                // WORKS with jdk1.1.8 FAILS with jdk1.3 and JVIEW ???

            szCurrSection += txt.length();  // add to total size

            // Have we overflowed the max?
            if (szCurrSection > maxSectionSz) {
                if (debug) {
                    System.out.println("Overflowed1: @ " + szCurrSection 
                                         + " vs "+ maxSectionSz 
                                         + " currSec=" + currSection 
                                         + " lastSecSvd=" + lastSectionSaved);
                }
                // Too big, save and start another one
                savedSections.addElement(super.getText());     // save
                super.setText("");      // clear the display
                szCurrSection = 0;      // reset
                currSection++;
                lastSectionSaved++;
                tellListener();         // tell listener there is a new section
                enableControls();
            }  // end overflow

        }else {
            // If not at current section, need to add to non-displayed section
            // NOTE: NEED to test if txt is TOO BIG to add??? - see breakUpStr()
            svText.append(txt);
            szCurrSection += txt.length();  // add to total size
            // Have we overflowed?
            if (szCurrSection > maxSectionSz) {
                if (debug) {
                    System.out.println("Overflowed2: @ " + szCurrSection 
                            + " vs " + maxSectionSz 
                            + " currSec=" + currSection 
                            + " lastSecSvd=" + lastSectionSaved);
                }
                // save current and move to next
                savedSections.addElement(svText.toString());
                svText.setLength(0);    // clear buffer
                szCurrSection = 0;      // reset
                lastSectionSaved++;          // have one more
                tellListener();         // tell listener there is a new section
                enableControls();
            } // end overflow
        }
    } // end append()

    //---------------------------------------------------------------------
    /** Set the text to a string
    */
    public synchronized void setText(String txt) {
        if (debug)
            System.out.println("setText() got: >>>> length="
                    + txt.length() + " maxSS=" + maxSectionSz);
        // Break txt up into chunks (if >max size) and save
        // If txt.length < max,  clear current settings and start fresh
        if (txt.length() < maxSectionSz) {
            super.setText(txt);
            // Clear or reset all ptrs etc
            svText.setLength(0);
            currSection = 0;
            szCurrSection = txt.length();   // save size
            savedSections.removeAllElements();  // empty the Vector
            lastSectionSaved = NONE_SAVED;
            enableControls();
            return;

        } else {
            // Larger than a section, split up into sections
            Vector secs = breakUpStr(txt, maxSectionSz);
            if (debug)
                System.out.println("Broken up to " + secs.size() + " pieces");
//                            + "\n" + secs.toString());

            // Empty the savedSections
            savedSections.removeAllElements();

            // Save 1 thru n-1
            for(int i = 0; i < secs.size()-1; i++)
                savedSections.addElement(secs.elementAt(i));

            // Last element goes in the TextArea or svText 
            // NOTE: Possible boundary problem here???
            super.setText((String)secs.lastElement());

            // Update pointers etc
            szCurrSection = ((String)secs.lastElement()).length();
            lastSectionSaved = secs.size() - 2;
            currSection = lastSectionSaved + 1;
            svText.setLength(0);
            tellListener();
            enableControls();
        }
    } // end setText()

    // If we're not on the first section, the SelectionStart and End need to be adjusted
    public String getSelectedText() {
        if (lastSectionSaved == NONE_SAVED      // simple if nothing saved
            || currSection == 0) {              // or in first section
            return super.getSelectedText(); 

        }else {
          // Need to find the section and get the text there
          String sel = "";
          if (currSection <= lastSectionSaved) {
            sel = ((String)savedSections.elementAt(currSection)).substring(
                           getSelectionStart(), getSelectionEnd());
          } else {
            // Interesting that getText() works, but getSelectedText() fails???
            // getText() gets from currently displayed, getSelectText() from old
            sel = super.getText().substring(
                           getSelectionStart(), getSelectionEnd());
          }
          if (debug) {
            System.out.println("getSelectedText: " + getSelectionStart() 
                           + " " + getSelectionEnd() + "\n >>>" + sel);
          }
          return sel;
       }
    }

    //-------------------------------------------------------
    // Local method for breaking a String up into Sections
    // Returns a Vector of Strings: 1st=enuff to fill sector, 2-n-1=sector, n=remndr
    private Vector breakUpStr(String str, int maxFirst) {
        Vector svSecs = new Vector();
        int start = 0;                  // for scanning

        // Scan to next NL, and put substring into Vector
        while (start < str.length()-1) {  
            int ix = findMaxNL(str, start, maxFirst);
//  System.out.println("breakUpStr: ix=" + ix + " s=" + start + " mF=" + maxFirst);
            if (ix < 0) {
                // No more NLs found
                if (start < str.length()) {
                    // Get last bit of str
                    svSecs.addElement(str.substring(start));
                }
                break;                  // done
            }  // no more nls
            ix++;                       // advance past nl
            svSecs.addElement(str.substring(start, ix)); // save a section
            start = ix;                 // set scan ptr
            maxFirst = maxSectionSz;    // now use max
        } // end while () loop

        return svSecs;
    }  // end breakUpStr()

    //-----------------------------------------------------
    // Method to find last nl before maxSectionSz
    // Returns index of nl.
    private int findMaxNL(String str, int start, int max) {
//        System.out.println("findMaxNL: " + str.length() + " s=" + start 
//            + " max=" + max + " str.len=" + str.length() 
//            + " nlChar= " + new Integer('\n').intValue() 
//            + "\n"+ showBytes(str.getBytes()));

        int     lastNL = -1;
        int     startLine = start;      //Save start of first rec

        while(true) {
            int ix = str.indexOf(nlChar, start);
//            System.out.println("ix=" + ix + " start=" + start);
            if (ix < 0 || (ix - startLine > max))
                return lastNL;          // Not found
            lastNL = ix;                // save nl's index
            start = ix+1;               // Advance over it
        } // end while(true) 
    }  // end findMaxNL()

    //------------------------------------------------------------------
    /** Get all the save text and return 
    */
    public synchronized String getText() {
        // NOTE: This routine is called by super.append()!!!
        if (lastSectionSaved == NONE_SAVED || doingSuperAppend)
            return super.getText();     // simple if nothing saved yet

        else {
            // Get text from all the save buffers
            StringBuffer sb = new StringBuffer((lastSectionSaved+2)*maxSectionSz);
            for (int i = 0; i <= lastSectionSaved; i++) {
                sb.append((String)savedSections.elementAt(i));
            }
            // Now check where last section is:  TextArea or svText
            if (currSection > lastSectionSaved)
                sb.append(super.getText());     // TextArea if at last section
            else
                sb.append(svText.toString());   // Else being saved in svText

            return sb.toString();
        }
    }  // end getText()

    //-------------------------------------------------------------
    /** Copy current contents of svText to a saved buffer and reset
    // NOT USED???
    */
    public synchronized void flush() {
        if (svText.length() > 0 ) {
            savedSections.addElement(svText.toString());
            svText.setLength(0);    // clear buffer
            szCurrSection = 0;      // reset
            lastSectionSaved++;     // have one more
            tellListener();         // tell listener there is a new section
            enableControls();
            System.out.println("Flushed section " + lastSectionSaved);
        } else if(szCurrSection > 0) {
            savedSections.addElement(super.getText());     // save
            lastSectionSaved++;     // have one more
            tellListener();         // tell listener there is a new section
            enableControls();
        }
    } // end flush()

    //-------------------------------------------------------------
    public Dimension getPreferredSizeXXX() {
        System.out.println("getPreferredSize: " + super.getPreferredSize()
                + " parent= " + getParent());
        return super.getPreferredSize();
    }
/*      Output from above:
  getPreferredSize: java.awt.Dimension[width=440,height=350] 
      parent= java.awt.Frame[frame0,0,0,140x32,invalid,hidden,
                 layout=java.awt.BorderLayout,resizable,title=Testing BigTextArea]
*/

    //-------------------------------------------------------------------
    // Process button pushes
    public void actionPerformed(ActionEvent ae){
        Object src = ae.getSource();
        String cmd = ((Button)src).getLabel();
        if (cmd.startsWith(svUpLbl)) {
            goUp();
        }else if (cmd.startsWith(svDownLbl)) {
            goDown();

        }else {
            System.err.println("Unknown ae " + ae + " src=" + ae.getSource());
        }
    } // end actionPerformed()


    //***************************************************************************
    // FOLLOWING For TESTING 
    // Be sure you are in the correct directory for the package above: NormsTools
/*    // comment out when done by adding /* at start of this line <<<<<<
    //--------------------------------------------------------------------------
    // Debugging routine to convert byte array to a hex string for display
    static char[] hexValue = {'0', '1', '2', '3', '4' , '5', '6', '7', '8', '9',
                                'A', 'B', 'C', 'D', 'E','F'};

    static String showBytes(byte[] rec) {
        StringBuffer sb = new StringBuffer((rec.length * 2 + (rec.length / 4)));
        sb.setLength(rec.length * 2 + (rec.length / 4));  // why do I need this???
        int j = 0;                      // Output pointer
      try {
        for(int i = 0; i < rec.length; i++ ) {
            int ix = rec[i] & 0xFF;
            sb.setCharAt(j++, hexValue[ix >> 4]);
            sb.setCharAt(j++, hexValue[ix & 0x0F]);
            if (i % 4 == 3)
                sb.setCharAt(j++, ' ');     // insert blank every 4th char
        }
      }catch(Exception ex) {
        System.err.println("showBytes() Error " + ex + "\n j=" + j + " " + sb.length());
      }
        return sb.toString();
    } // end showBytes()

    // ************************************************************
    //              S T A R T    H E R E
    // ************************************************************
    public static void main(String[] args) {
        new NormsTools.BigTextAreaTester();
    }
} // end class   For Testing purposes (skipped when above commented out)


    //-------------------------------------------------------------------
    // An local class for testing 
    class BigTextAreaTester implements BigTextAreaListener, ActionListener {

        private  BigTextArea bta;
        Frame   f;

        Button upB = new Button("Up");
        Button downB = new Button("Down");
        Button addMoreB = new Button("Add more");
        Button findB = new Button("Find");
        Button replaceB = new Button("Replace");
        Button getSelB = new Button("Get Sel");


        // Constructor
        BigTextAreaTester() {
            f = new Frame("Testing BigTextArea");
            f.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent we) {
                    f.dispose();
                    System.exit(0);
                }
            });
            f.setLayout(new BorderLayout());
            f.setBackground(new Color(255, 0, 120));
            Panel pTop = new Panel();
            pTop.add(new Label("North"));
            f.add(pTop, BorderLayout.NORTH);
    
            bta = new BigTextArea(22, 60, 20000);
            f.add(bta, BorderLayout.CENTER);
    
            Panel pBot = new Panel();
            upB.addKeyListener(new MakeEnterDoAction());
            downB.addKeyListener(new MakeEnterDoAction());
            pBot.add(upB);
            pBot.add(downB);

            // Give button management to bta 
            bta.setButtons(upB, downB, true);
            bta.addBigTextAreaListener(this);
            addMoreB.addActionListener(this);
            pBot.add(addMoreB);
            // Add a button to simulate Find to end of text
            findB.addActionListener(this);
            pBot.add(findB);
            replaceB.addActionListener(this);
            pBot.add(replaceB);
            getSelB.addActionListener(this);
            pBot.add(getSelB);
            f.add(pBot, BorderLayout.SOUTH);
    
            f.pack();
            f.show();
    
            // Now generate output and add to 
            for (int i = 0; i < 1200; i+=2) {
                try{
                    bta.append(
          "Adding line " + i + " that is the first of two lines on the display.\n");
                    bta.append("Adding line " + (i+1) 
    + " that is longer than the other by some amount and that will fill the BTA\n");
//                    System.out.println("Added line " + i);
                }catch(Exception ex){
                    System.out.println("Ex in add loop at " + i);
                    ex.printStackTrace();
                    break;      // EXIT
                }
            }
            // Test select()
//            bta.select(bta.getMaxSectionSize()+10, bta.getMaxSectionSize()+15);

        } // end Constructor

        // Test listener interface method:
        public void addedSection(int section) {
            System.out.println("Listener: Added section " + section);
        }

        private void addMore(String str) {
            for (int i = 0; i < 30; i++) {
                try{Thread.sleep(10);}catch(Exception e){}
                bta.append("Adding more lines " + i + str 
                            + " on the end of the display to see what hapens.\n");
            }
        } // end addMore()
    
        private void replaceIt() {
            // Get the text and put it back with a flag
            String recs = bta.getText();
            System.out.println("replaceIt(): Got text: " + recs.length() + " bytes");
            bta.setText(">>> Replaced lines at " + new Date() + "\n" + recs
                        + "*** End replaced lines <<<\n");
        }

        // Trap button pushes
        public void actionPerformed(ActionEvent ae) {
            Object src = ae.getSource();
            String cmd = ((Button)src).getLabel();

            if(cmd.equals("Add more")){
                addMore(" at " + (new Date()).toString());

            }else if (cmd.equals("Find")) {
                String txt = bta.getText();
                bta.select(txt.length()-5, txt.length()-1);

            }else if(cmd.equals("Replace")) {
                replaceIt();

            }else if(src == getSelB) {
                System.out.println("Selected: " + bta.getSelectedText());

            }else {
                System.out.println("Unknown ae " + ae + " src=" + ae.getSource());
            }
        } // end actionPerformed()

    // end of testing code */
}  // end class BigTextArea
